# Copyright (C) 2025 PHYTEC Messtechnik GmbH
# Author: Wadim Egorov <w.egorov@phytec.de>
#
# This recipe provides U-Boot scripts for booting Ampliphy. It is intended
# for use with U-Boot's Standard Boot framework:
#
#   https://docs.u-boot.org/en/latest/develop/bootstd/overview.html
#
# In order for your board to be able to boot Ampliphy, you have to make sure
# your board provides all neccessary variables. The scripts generated by
# amliphy-boot.bb makes use of the environment variable conventions
# documented in:
#
#   https://docs.u-boot.org/en/latest/develop/bootstd/overview.html#environment-variables
#
# The final boot script file is generated with mkimage and packed into a FIT.
#
#
# Other Environment Variables used:
#
# fit_addr_r (optional)
#   In case if the default $loadaddr is not suitabale for loading a fitImage
# overlaysenvfile (optional)
#   Name of the env file on the boot partition containing default overlays if
#   default "overlays.txt" is not suitabale.

DESCRIPTION = "This recipe compiles U-Boot scripts into FIT images, which are \
loaded by U-Boot using the standard boot script method. These scripts provide \
a consistent boot process for the Ampliphy distribution, ensuring compatibility \
across different platforms."
SUMMARY = "U-Boot scripts for booting Ampliphy"
LICENSE = "MIT"
LIC_FILES_CHKSUM = "file://${COMMON_LICENSE_DIR}/MIT;md5=0835ade698e0bcf8506ecda2f7b4f302"

DEPENDS = "u-boot-mkimage-native dtc-native"

SRC_URI = " \
    file://mmc_boot.cmd \
    file://spi_boot_fit.cmd.in \
    file://mmc_boot_fit.cmd \
    file://net_boot_fit.cmd.in \
    file://boot.its.in \
"

inherit deploy

MMC_BOOT_SCRIPT ?= "mmc_boot.cmd"
MMC_BOOT_SCRIPT:secureboot ?= "mmc_boot_fit.cmd"

# Used by the spi boot script to locate the fitImage
SPI_MTD_PARTS ?= ""
SPI_MTD_PARTS:k3 ?= "nor0:-@0x740000(fitimage)"

# Used by the net boot script to pass the kernel ip parameter
IP_PARAM ?= ""
IP_PARAM:k3 ?= "dhcp"

# Used by the net boot script to load the image from network
NET_FETCH_CMD ?= ""
NET_FETCH_CMD:k3 ?= "dhcp"

S = "${WORKDIR}"

do_compile() {
    sed -e 's/@@SPI_MTD_PARTS@@/${SPI_MTD_PARTS}/' "${WORKDIR}/spi_boot_fit.cmd.in" > spi_boot_fit.cmd
    sed -e 's/@@IP_PARAM@@/${IP_PARAM}/' \
        -e 's/@@NET_FETCH_CMD@@/${NET_FETCH_CMD}/' "${WORKDIR}/net_boot_fit.cmd.in" > net_boot_fit.cmd

    for script in *.cmd ; do
        sed -e "s/@@BOOTCOMMAND_FILE@@/${script}/" "${WORKDIR}/boot.its.in" > boot.its
        mkimage -C none -A ${UBOOT_ARCH} -f boot.its ${script}.scr.uimg
    done
}

do_deploy() {
    install -d ${DEPLOYDIR}

    for cmdimg in *.uimg ; do
        img=$(echo "$cmdimg" | sed 's/.cmd//')  # drop .cmd
        install -m 0644 ${cmdimg} "${DEPLOYDIR}/${img}"
    done

    install -m 0644 ${S}/${MMC_BOOT_SCRIPT}.scr.uimg ${DEPLOYDIR}/boot.scr.uimg
}

addtask deploy after do_install before do_build
