#!/bin/bash
#
# Release script for demo-welcome-webpage
# Creates a tar.gz archive with the format: <project>_<version>.tar.gz
# and creates/pushes a git tag
#
# Usage:
#   ./scripts/release.sh major          # Increment major version (v1.0 -> v2.0)
#   ./scripts/release.sh minor          # Increment minor version (v1.0 -> v1.1)
#
#   Options:
#   --try-run - Try to create the archive and tag locally, but don't push to remote


set -euo pipefail

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Script directory and project root
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "${SCRIPT_DIR}/.." && pwd)"
PROJECT_NAME="demo-welcome-webpage"

# Function to print error messages
error() {
    echo -e "${RED}Error:${NC} $1" >&2
    exit 1
}

# Function to print info messages
info() {
    echo -e "${GREEN}Info:${NC} $1"
}

# Function to print warning messages
warning() {
    echo -e "${YELLOW}Warning:${NC} $1"
}

# Function to show usage
usage() {
    echo "Usage: $0 {major|minor} [--try-run]"
    echo ""
    echo "  major     - Increment major version (e.g., v1.0 -> v2.0)"
    echo "  minor     - Increment minor version (e.g., v1.0 -> v1.1)"
    echo "  --try-run - Create archive and tag locally, but don't push to remote"
    exit 1
}

# Check if we're in a git repository
if ! git rev-parse --git-dir > /dev/null 2>&1; then
    error "Not in a git repository. Please run this script from within the project directory."
fi

# Parse command line arguments
TRY_RUN=false
RELEASE_TYPE=""

for arg in "$@"; do
    case "$arg" in
        --try-run)
            TRY_RUN=true
            ;;
        major|minor)
            if [ -n "$RELEASE_TYPE" ]; then
                error "Release type specified multiple times. Use 'major' or 'minor' only once."
            fi
            RELEASE_TYPE="$arg"
            ;;
        *)
            error "Unknown argument: '$arg'. Use 'major', 'minor', or '--try-run'."
            ;;
    esac
done

# Validate release type was provided
if [ -z "$RELEASE_TYPE" ]; then
    error "Release type is required. Use 'major' or 'minor'."
fi

if [ "$TRY_RUN" = true ]; then
    info "Try-run mode: Tag will be created locally but not pushed to remote"
fi

# Get the latest git tag
LATEST_TAG=""
if git describe --tags --abbrev=0 > /dev/null 2>&1; then
    LATEST_TAG=$(git describe --tags --abbrev=0)
    info "Latest tag found: $LATEST_TAG"
else
    error "No git tags found. Please create an initial tag first (e.g., git tag v0.1)."
fi

# Validate tag format (should start with 'v' and be vX.Y)
if [[ ! "$LATEST_TAG" =~ ^v[0-9]+\.[0-9]+(-.*)?$ ]]; then
    error "Invalid version format in tag '$LATEST_TAG'. Expected format: vX.Y (e.g., v1.0)"
fi

# Extract version number from tag (remove 'v' prefix for parsing)
OLD_VERSION="${LATEST_TAG#v}"

# Parse version components (X.Y format)
IFS='.' read -ra VERSION_PARTS <<< "${OLD_VERSION%%-*}"
MAJOR="${VERSION_PARTS[0]:-0}"
MINOR="${VERSION_PARTS[1]:-0}"

# Validate we have both major and minor
if [ -z "$MAJOR" ] || [ -z "$MINOR" ]; then
    error "Invalid version format in tag '$LATEST_TAG'. Expected format: vX.Y"
fi

# Increment version based on release type
if [ "$RELEASE_TYPE" == "major" ]; then
    MAJOR=$((MAJOR + 1))
    MINOR=0
    info "Incrementing major version"
else
    MINOR=$((MINOR + 1))
    info "Incrementing minor version"
fi

# Construct new version (with 'v' prefix for tag, without for filename)
NEW_VERSION_TAG="v${MAJOR}.${MINOR}"
NEW_VERSION_FILE="${MAJOR}.${MINOR}"

info "Version: $LATEST_TAG -> $NEW_VERSION_TAG"

# Check if there are uncommitted changes
if ! git diff-index --quiet HEAD --; then
    warning "You have uncommitted changes. They will not be included in the archive."
    read -p "Continue anyway? (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        info "Aborted."
        exit 0
    fi
fi

# Create output filename (without 'v' prefix)
OUTPUT_FILE="${PROJECT_NAME}_${NEW_VERSION_FILE}.tar.gz"
OUTPUT_PATH="${PROJECT_ROOT}/${OUTPUT_FILE}"

# Check if output file already exists
if [ -f "$OUTPUT_PATH" ]; then
    warning "Output file already exists: $OUTPUT_FILE"
    read -p "Overwrite? (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        info "Aborted."
        exit 0
    fi
    rm -f "$OUTPUT_PATH"
fi

# Check if tag already exists
if git rev-parse "$NEW_VERSION_TAG" > /dev/null 2>&1; then
    error "Tag '$NEW_VERSION_TAG' already exists. Please delete it first or use a different version."
fi

# Create the archive
info "Creating archive: $OUTPUT_FILE"
cd "$PROJECT_ROOT"

if git archive --format tar.gz --prefix="${PROJECT_NAME}-${NEW_VERSION_FILE}/" -o "$OUTPUT_PATH" HEAD; then
    # Verify the archive was created
    if [ -f "$OUTPUT_PATH" ]; then
        ARCHIVE_SIZE=$(du -h "$OUTPUT_PATH" | cut -f1)
        info "Archive created successfully: $OUTPUT_FILE ($ARCHIVE_SIZE)"
    else
        error "Archive creation failed - file not found."
    fi
else
    error "Failed to create archive."
fi

# Create git tag
info "Creating git tag: $NEW_VERSION_TAG"
if git tag -a "$NEW_VERSION_TAG" -m "Release version $NEW_VERSION_TAG"; then
    info "Tag created successfully: $NEW_VERSION_TAG"
else
    error "Failed to create git tag."
fi

# Push tag to remote (skip in try-run mode)
if [ "$TRY_RUN" = true ]; then
    warning "Try-run mode: Skipping push to remote"
    info "To push the tag manually, run: git push origin $NEW_VERSION_TAG"
else
    info "Pushing tag to remote..."
    if git push origin "$NEW_VERSION_TAG"; then
        info "Tag pushed successfully to remote."
    else
        warning "Failed to push tag to remote. You may need to push manually: git push origin $NEW_VERSION_TAG"
    fi
fi

info "Release complete!"
info "  Archive: $OUTPUT_FILE"
info "  Tag: $NEW_VERSION_TAG"
if [ "$TRY_RUN" = true ]; then
    info "  Mode: Try-run (tag not pushed)"
fi
info "  Location: $OUTPUT_PATH"
